package connect4;

import info.gridworld.grid.BoundedGrid;
import info.gridworld.grid.Location;
import info.gridworld.world.World;

public class GameBoard extends World<Disc>
{
	private Player player1; // blue
	private Player player2; // red
	private boolean turn; // true is blue, false is red
	private boolean gameOver;
	
	private BoardUtilities boardUtils; // methods useful to both GameBoard and AI players
	
	/**
	 * Constructs a GameBoard for 2 human players.
	 */
	public GameBoard()
	{
		super(new BoundedGrid<Disc>(6, 7));
		boardUtils = new BoardUtilities((BoundedGrid<Disc>) getGrid());
		player1 = null;
		player2 = null;
		start();
	}
	
	/**
	 * Constructs a GameBoard for 1 human player and a computer opponent.
	 * @param p1 the computer opponent
	 */
	public GameBoard(Player p1)
	{
		this();
		player1 = p1;
	}
	
	/**
	 * Constructs a GameBoard for 2 computer players
	 * @param p1 a computer player
	 * @param p2 a computer player
	 */
	public GameBoard(Player p1, Player p2)
	{
		this();
		player1 = p1;
		player2 = p2;
	}
	
	/**
	 * Starts or restarts game.
	 */
	public void start()
	{
		for(Location disc : getGrid().getOccupiedLocations())
			getGrid().remove(disc);
		
		setMessage("");
		turn = true;
		gameOver = false;
	}
	
	/**
	 * When a location is clicked and it is a player's turn who is not
	 * controlled by an AI, this method will attempt to place a disc in 
	 * the appropriate column. If the location is already full,
	 * use setMessage to indicate an invalid location and do not
	 * add a disc anywhere.
	 * @param loc The location clicked.
	 * @return Should always consume the click and return true.
	 */
	public boolean locationClicked(Location loc)
	{
		setMessage("");
		
		if(gameOver)
		{
			start();
			return true;
		}
		
		if( (turn && player1 != null) || (!turn && player2 != null) ) // AI's turn, not players
		{
			setMessage("Computer's turn.\nPress Step for computer's move.");
			return true;
		}
		
		int row = boardUtils.firstEmptyRow(loc.getCol());
		
		if(row == -1) 
			setMessage("Column " + loc.getCol() + " is full.\nChoose a different column.");
		else
			addDisc(loc.getCol());
		
		return true;
	}
	
	/**
	 * Takes the game through one turn.
	 */
	public void step()
	{
		setMessage("");
		
		if(gameOver)
		{
			setMessage("Game is over.\nClick to reset.");
			return;
		}

		//Player1's Turn
		if(turn)
		{
			//Player 1 is an AI agent
			if(player1 != null)
			{
				//Update the board for both players (for AI purposes)
				player1.setBoard((BoundedGrid<Disc>)getGrid());
				addDisc(player1.move());
			}
			else
				setMessage("Blue's turn.\nClick a column to drop your piece.");
		}
		else //Player2's turn
		{
			if(player2 != null)
			{
				player2.setBoard((BoundedGrid<Disc>)getGrid());
				addDisc(player2.move());
			}
			else
				setMessage("Red's turn.\nClick a column to drop your piece.");

		}
	}

	/**
	 * Adds appropriately colored disc to the specified column. Then      	
	 * checks to see if the game is over. If it's not, updates the turn 
	 * and sets the world's message accordingly.
	 * @param col The column to add the disc to.
	 */
	public void addDisc(int col)
	{
		setMessage("");
		
		int row = boardUtils.firstEmptyRow(col);
		
		if(row == -1)
			throw new IllegalArgumentException("Column " + col + " is full");
		
		if(boardUtils.wouldWin(turn, col))
		{
			setMessage((turn ? "Blue" : "Red") + " wins!\nClick to restart.");
			getGrid().put(new Location(row, col), new Disc(turn));
			gameOver = true;
		}
		else
		{
			getGrid().put(new Location(row, col), new Disc(turn));
			turn = ! turn;
		}
		
		if(!gameOver && getGrid().getOccupiedLocations().size() == getGrid().getNumRows() * getGrid().getNumCols())
		{
			setMessage("Game is a draw.\nClick to restart.");
			gameOver = true;
		}
	}
}